import numpy as np
import torch

def sample_directions(n_samples: int, d: int) -> np.ndarray:
    """
    Deterministically sample points on the (d-1)-sphere S^{d-1} using generalized spherical coordinates.

    Parameters:
        n_samples (int): Number of samples to generate.
        d (int): Dimension of the ambient space R^d (so the sphere is S^{d-1}).

    Returns:
        np.ndarray: (n_samples, d) array of points on the unit sphere.
    """
    if d < 2:
        raise ValueError("Dimension must be at least 2 to define a sphere.")

    # Create linspace angles
    angles = np.linspace(0, 2 * np.pi, n_samples, endpoint=False)
    
    samples = []

    for theta in angles:
        coords = [np.cos(theta)]  # x1 = cos(theta)
        sin_prod = np.sin(theta)  # sin(theta)

        # Use fixed angles for the rest, e.g., π/4 (could be made more sophisticated)
        for i in range(1, d - 1):
            phi = np.pi / 4
            coords.append(sin_prod * np.cos(phi))
            sin_prod *= np.sin(phi)

        coords.append(sin_prod)  # last coordinate
        samples.append(coords)

    return torch.Tensor(samples).to(torch.float32)